package com.anychart;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;

/**
 * This class stores the AnyChart JSP component settings and data:
 * javascript and swf code.
 *
 * @see "http://www.anychart.com"
 */
public final class AnyChart {

	/**
	 * Base identity string to create keys for any objects identification
	 * over any kind of contexts
	 */
	public static final String IDENTITY_STRING = "com/anychart/chart";
	
	/**
	 * Url part to AnyChart.swf content
	 */
	public static final String SWF_PATH = "com.anychart/swf/AnyChart.swf";
	
	/**
	 * Url part to AnyChart.js content
	 */
	public static final String JS_PATH = "com.anychart/js/AnyChart.js";
	
	/**
	 * Singleton instance of AnyChart class
	 */
	private static AnyChart instance = null;
	
	/**
	 * AnyChart class instance getting method
	 * @return AnyChart instance
	 */
	public static synchronized AnyChart getInstance() {
		if (instance == null) {
			instance = new AnyChart();
		}
		return instance;
	}
	
	/**
	 * internal class for store resource in memory
	 */
	public static class Resource {
		/**
		 * Resource content byte array
		 */
		private byte[] content = null;
		/**
		 * Resource last modified timestamp
		 */
		private long lastModified = 0L;
		
		/**
		 * Method writes resource content into {@link java.io.OutputStream}
		 * @param os {@link java.io.OutputStream} to write content to @throws IOException
		 * @throws IOException object
		 */
		public void writeContent(OutputStream os) throws IOException {
			os.write(this.content);
		}
		
		/**
		 * Resource last modified timestamp getting method
		 * @return milisecond number since epoch
		 */
		public long getLastModified() {
			return this.lastModified;
		}
	}
	
	/**
	 * field to store AnyChart.js
	 */
	private Resource js;
	
	/**
	 * field to store AnyChart.swf
	 */
	private Resource swf;
	
	/**
	 * private constructor required by singleton pattern
	 */
	private AnyChart() {
		this.init();
	}

	/**
	 * Init instance method
	 */
	private void init() {
		this.js = this.loadResource("resources/AnyChart.js");
		this.swf = this.loadResource("resources/AnyChart.swf");
	}
	
	/**
	 * Method loads internal resource into memory,
	 * creates {@link Resource} object and return it
	 * 
	 * @param path relative file path to internal resource 
	 * @return {@link Resource} class instance 
	 */
	private Resource loadResource(String path) {
		URL url = this.getClass().getResource(path);
		if (url == null) {
			throw new RuntimeException("Can't find " + path);
		}
		Resource res = new Resource();
		InputStream is = null;
		try {
			URLConnection conn = url.openConnection();
			res.lastModified = conn.getLastModified();
			is = conn.getInputStream();
			res.content = new byte[is.available()];
			for (int i = 0;
				i < res.content.length;
				i += is.read(res.content, i, res.content.length - i)
			);
		} catch (IOException e) {
			throw new RuntimeException(path + " load failed", e);
		} finally {
			try {
				if (is != null) {
					is.close();
				}
			} catch (Exception ignore) {}
		}
		return res;
	}

	/**
	 * @return the js resource
	 */
	public Resource getJs() {
		return js;
	}

	/**
	 * @return the swf resource
	 */
	public Resource getSwf() {
		return swf;
	}
	
}